/** @file   mp3file.h
 * @brief   Declaration of MP3File class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */


#ifndef H_ENG2D_MP3FILE_H
#define H_ENG2D_MP3FILE_H

#include <allegro.h>
#include <string>
#include "eng2d_dll.h"

// Forward declarations
struct ALMP3_MP3;

namespace eng2d {


/** @class  MP3File
 * @brief   Represents a mp3 file that can be given to MP3Player to be played.
 * @author  Tomi Lamminsaari
 *
 * To use mp3-music in your programs you should first create an instance of
 * this MP3File - class. Then you can pass that object to the MP3Player to
 * be played.
 *
 * As the MP3Player starts playing this file (or you call the open() - method
 * by yourself) the whole mp3-file will be loaded into the memory. So be
 * careful not to play too large mp3-files.
 *
 * This class uses the AllegroMP3-library to manage the mp3-files.
 *
 */
class DLLIMPORT MP3File
{
public:

  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Creates new MP3File and attaches it to a given file.
   * @param     file              Name of the mp3-file you want this object
   *                              to represent.
   */
  MP3File( const std::string& file );
  
  
  /** Destructor.
   */
  virtual ~MP3File();



  ///
  /// Public methods
  /// ==============

  
  /** Opens the mp3-file. This method loads it entirely to the memory. You
   * shouldn't call this by yourself. The MP3Player-class handles the opening
   * of the mp3-files.
   * @return    Size of the mp3-file.
   */
  long open();
  
  /** Closes the mp3-file. Releases the memory that was allocated for it.
   */
  void close();


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the MP3-file for the mp3-player.
   * @return    The mp3-data.
   */
  ALMP3_MP3*  getMP3() const;
  
private:
  
  /** A copy constructor.
   */
  MP3File( const MP3File& rO );
  
  /** Operator =
   */
  MP3File& operator = (const MP3File& rO);
  

  ///
  /// Members
  /// =======

  /** THe name of the mp3-file. */
  std::string m_filename;
  
  /** Size of the loaded mp3-file. */
  long        m_filesize;
  
  /** Pointer to mp3-data got from AllegroMP3-library. */
  ALMP3_MP3*  m_pMP3;
  
  /** The mp3 raw data. */
  char*       m_pData;
  
  /** Did the mp3-file really open. */
  bool        m_open;
  
  
  friend class MP3Player;
};

};

#endif
